package ktbyte.gui;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import def.processing.core.PApplet;
import def.processing.core.PConstants;
import def.processing.event.KeyEvent;
import def.processing.event.MouseEvent;

/**********************************************************************************************************************
 * This is the <i>core</i> class of the KTGUI library. It is specifically designed to provide the ability of separating 
 * the 'main' code of the application from the 'GUI' related code. 
 * This class is used to 'transfer' the 'draw', 'mouse' and 'keyboard' events from PApplet to KTGUI components 
 * (controllers).
 * The main idea is to eliminate the need of adding the callback methods directly in the Processing's 'mouseXXXXX()' 
 * and 'keyXXXXX()' methods.
 * This class is used also as a factory to create the KTGUI components (controllers). 
 * When some event is received from PApplet, this class automatically iterates through all the components in the 
 * components list and 'transfers' the events to each component.
 * 
 *
<h4>The concept of &#39;main&#39; and &#39;GUI&#39; related code separation</h4>
<p>In order to separate the two types of the said codes, we can use the ability of the Processing library to &#39;register&#39; 
some of its &#39;specific&#39; methods in the external class. To register the &#39;specific&#39; methods in the external class, 
the <em>PApplet</em> uses the <code>registerMethod(String methodName, PApplet parentPApplet)</code> method. After these methods 
were being registered, they will be executed at a particular moments of the &#39;main&#39; code execution allowing to 
&#39;synchronize&#39; the execution of the &#39;main&#39; code with the execution of the particular methods of the external class. 
(More detailed information can be found in the Processing&#39;s 
documentation <a href='https://github.com/processing/processing/wiki/Library-Basics#library-methods'>here</a>).</p>
<p>The full list of these &#39;specific&#39; methods can be found 
<a href='https://github.com/processing/processing/wiki/Library-Basics#library-methods'>here</a>. We will be registering (in the 
external class) and using for separating the code <strong><em>only the following three methods</em></strong>:</p>
<ul>
<li><code>public void draw()</code> Method that&#39;s called at the end of  the <em>PApplet&#39;s</em> <code>draw()</code> method.</li>
<li><code>public void mouseEvent(MouseEvent e)</code> Method that&#39;s called when a mouse event occurs in the parent <em>PApplet</em>. 
Drawing inside this method is allowed because mouse events are queued, unless the sketch has called <code>noLoop()</code>.</li>
<li><code>public void keyEvent(KeyEvent e)</code> Method that&#39;s called when a key event occurs in the parent <em>PApplet.</em> 
Drawing is allowed because key events are queued, unless the sketch has called <code>noLoop()</code>.</li>
</ul>
 *********************************************************************************************************************/
public class KTGUI implements PConstants {
    private static PApplet               pa;
    private HashMap<Controller, Integer> garbageList;
    public ArrayList<String>             drawCallStack      = new ArrayList<String>();

    public static int                    COLOR_FG_HOVERED;
    public static int                    COLOR_FG_PRESSED;
    public static int                    COLOR_FG_PASSIVE;
    public static int                    COLOR_BG_HOVERED;
    public static int                    COLOR_BG_PASSIVE;
    public static int                    COLOR_BG_PRESSED;
    public static int                    DEFAULT_COMPONENT_WIDTH;
    public static int                    DEFAULT_ALIGN_GAP;
    public static int                    DEFAULT_ROUNDING;

    private static boolean               debugControllers   = false;
    private boolean                      debugDrawCallStack = false;

    /*************************************************************************************************************************
     * This is a constructor of the KTGUI class.
     * It automatically registers the 'draw', 'mouseEvent' and 'keyEvent' methods of this class in PApplet.
     ************************************************************************************************************************/
    public KTGUI(PApplet pa) {
        debug("\nCreating of the KTGUI instance started.");
        init(pa);
        debug("Creating of the KTGUI instance completed.\n");
    }

    private void init(PApplet pa) {
        debug("\tInitializing of the KTGUI started.");
        KTGUI.pa = pa;
        KTGUI.pa.registerMethod("draw", this);
        debug("\t\t'draw()' method has been registered in parent PApplet.");
        KTGUI.pa.registerMethod("mouseEvent", this);
        debug("\t\t'mouseEvent()' method has been registered in parent PApplet.");
        KTGUI.pa.registerMethod("keyEvent", this);
        debug("\t\t'keyEvent()' method has been registered in parent PApplet.");

        garbageList = new HashMap<Controller, Integer>();
        debug("\t\tGarbage list created.");

        COLOR_FG_PASSIVE = pa.color(150, 180, 150);
        COLOR_FG_HOVERED = pa.color(150, 220, 150);
        COLOR_FG_PRESSED = pa.color(110, 200, 110);
        COLOR_BG_PASSIVE = pa.color(190);
        COLOR_BG_HOVERED = pa.color(220);
        COLOR_BG_PRESSED = pa.color(210);

        DEFAULT_COMPONENT_WIDTH = 16;
        DEFAULT_ALIGN_GAP = 20;
        DEFAULT_ROUNDING = 8;
        debug("\t\tColor and size constants initialized.");

        // Call this method once in order to create the singleton StageManager instance now,
        // and not later when it would be needed
        StageManager.getInstance();

        debug("\tInitializing of the KTGUI completed.");
    }

    public static PApplet getParentPApplet() {
        return pa;
    }

    /*************************************************************************************************************************
     * This method is intended to be called <b>automatically</b> at the end of each draw() cycle of the parent PApplet. 
     * This way, the KTGUI class automatically updates all the controllers on the <i>default</i> and <i>active</i> stages.
     ************************************************************************************************************************/
    public void draw() {
        if (StageManager.getInstance().userStagesExist()) {
            StageManager.getInstance().getActiveStage().draw();
        }
        StageManager.getInstance().getDefaultStage().draw();

        collectGarbage();
    }

    public void setDrawCallStackFlag(boolean debug) {
        this.debugDrawCallStack = debug;
    }

    void drawDebugTextSplitLine(int x, int y) {
        pa.text("--------------------------------------------------------------" +
                "--------------------------------------------------------------",
                x, y);
    }

    public void addDrawCallStackDebugMessage(String msg) {
        if (debugDrawCallStack) {
            drawCallStack.add(msg);
        }
    }

    public void addToGarbage(Controller controller, int millis) {
        garbageList.put(controller, millis);
    }

    @SuppressWarnings("rawtypes")
    void collectGarbage() {
        for (Map.Entry me : garbageList.entrySet()) {
            Controller controller = (Controller) me.getKey();
            int time = (Integer) me.getValue();
            if (pa.millis() - time > 100) {
                if (controller.parentStage != null) {
                    controller.parentStage.unregisterController(controller);
                } else {
                    if (controller.parentController != null) {
                        controller.parentController.detachController(controller);
                    }
                }
            }
        }
    }

    //-------------------------------------------------------------------------------------------------------------------
    // These are the 'factory' methods
    //-------------------------------------------------------------------------------------------------------------------

    public Button createButton(String title, int x, int y, int w, int h) {
        return new Button(this, title, x, y, w, h);
    }

    public Button createButton(int x, int y, int w, int h) {
        return new Button(this, "A Button", x, y, w, h);
    }

    public ArrowButton createDirectionButton(String title, int x, int y, int w, int h, int dir) {
        return new ArrowButton(this, title, x, y, w, h, dir);
    }

    public ArrowButton createDirectionButton(int x, int y, int w, int h, int dir) {
        return new ArrowButton(this, "A DirButton", x, y, w, h, dir);
    }

    public Slider createSlider(String title, int posx, int posy, int w, int h, int sr, int er) {
        return new Slider(this, title, posx, posy, w, h, sr, er);
    }

    public Slider createSlider(int posx, int posy, int w, int h, int sr, int er) {
        return new Slider(this, "A Slider", posx, posy, w, h, sr, er);
    }

    public Panel createPanel(String title, int x, int y, int w, int h) {
        Panel panel = new Panel(this, title, x, y, w, h);
        return panel;
    }

    public Panel createPanel(int x, int y, int w, int h) {
        Panel panel = new Panel(this, "A Panel", x, y, w, h);
        return panel;
    }

    public Pane createPane(String title, int x, int y, int w, int h) {
        Pane pane = new Pane(this, title, x, y, w, h);
        return pane;
    }

    public Pane createPane(int x, int y, int w, int h) {
        Pane pane = new Pane(this, "A Pane", x, y, w, h);
        return pane;
    }

    public ScrollBar createScrollBar(int x, int y, int w, int h, int sr, int er) {
        ScrollBar scrollBar = createScrollBar("A ScrollBar", x, y, w, h, sr, er);
        return scrollBar;
    }

    public ScrollBar createScrollBar(String title, int x, int y, int w, int h, int sr, int er) {
        if (w > h) {
            if ((w - 2 * h) < 2 * KTGUI.DEFAULT_COMPONENT_WIDTH) {
                debug("ERROR: The width of the ScrollBar to be created is "
                        + " too small. As a consequence, the internal slider would have "
                        + " orthogonal direction. Cannot create ScrollBar. Returning "
                        + "null reference.");
                return null;
            }
        } else {
            if ((h - 2 * w) < 2 * KTGUI.DEFAULT_COMPONENT_WIDTH) {
                debug("ERROR: The height of the ScrollBar to be created is "
                        + " too small. As a consequence, the internal slider would have "
                        + " orthogonal direction. Cannot create ScrollBar. Returning "
                        + "null reference.");
                return null;
            }
        }

        ScrollBar scrollBar = new ScrollBar(this, title, x, y, w, h, sr, er);
        return scrollBar;
    }

    public InputTextBox createInputTextBox(String title, int x, int y, int w, int h) {
        return new InputTextBox(this, title, x, y, w, h);
    }

    public InputTextBox createInputTextBox(int x, int y, int w, int h) {
        return createInputTextBox("An InputTextBox", x, y, w, h);
    }

    /**
     * This method 'redirects' the emitted mouse event from PApplet to KTGUI 'transfer' methods.
     * This method will be called <b>automatically</b> when the PApplet.mouseEvent is happening.
     */
    public void mouseEvent(MouseEvent e) {
        switch (e.getAction()) {
        case MouseEvent.PRESS:
            this.mousePressed();
            break;
        case MouseEvent.RELEASE:
            this.mouseReleased();
            break;
        case MouseEvent.DRAG:
            this.mouseDragged();
            break;
        case MouseEvent.MOVE:
            this.mouseMoved();
            break;
        case MouseEvent.WHEEL:
            this.mouseWheel(e);
            break;
        }
    }

    /**
     * This method 'redirects' the emitted keyboard event from PApplet to KTGUI 'transfer' methods.
     * This method will be called <b>automatically</b> when the PApplet.keyEvent is happening.
     */
    public void keyEvent(KeyEvent e) {
        switch (e.getAction()) {
        case KeyEvent.PRESS:
            this.keyPressed();
            break;
        case KeyEvent.RELEASE:
            this.keyReleased();
            break;
        }
    }

    public static void debug(String string) {
        if (debugControllers)
            PApplet.println(string);
    }

    /**
     * This is a 'transfer' method - it 'redirects' the PApplet.mouseDragged event to KTGUI components (controllers)
     */
    private void mouseDragged() {
        for (Controller controller : StageManager.getInstance().getActiveStage().controllers) {
            controller.processMouseDragged();
        }
        if (StageManager.getInstance().getDefaultStage() != StageManager.getInstance().getActiveStage()) {
            for (Controller controller : StageManager.getInstance().getDefaultStage().controllers) {
                controller.processMouseDragged();
            }
        }
    }

    /**
     * This is a 'transfer' method - it 'redirects' the PApplet.mousePressed event to KTGUI components (controllers)
     */
    private void mousePressed() {
        for (Controller controller : StageManager.getInstance().getActiveStage().controllers) {
            controller.processMousePressed();
        }

        if (StageManager.getInstance().getDefaultStage() != StageManager.getInstance().getActiveStage()) {
            for (Controller controller : StageManager.getInstance().getDefaultStage().controllers) {
                controller.processMousePressed();
            }
        }
    }

    /**
     * This is a 'transfer' method - it 'redirects' the PApplet.mouseReleased event to KTGUI components (controllers)
     */
    private void mouseReleased() {
        for (Controller controller : StageManager.getInstance().getActiveStage().controllers) {
            controller.processMouseReleased();
        }
        if (StageManager.getInstance().getDefaultStage() != StageManager.getInstance().getActiveStage()) {
            for (Controller controller : StageManager.getInstance().getDefaultStage().controllers) {
                controller.processMouseReleased();
            }
        }
    }

    /***
     * This is a 'transfer' method - it 'redirects' the PApplet.mouseMoved event to KTGUI components (controllers)
     */
    private void mouseMoved() {
        for (Controller controller : StageManager.getInstance().getActiveStage().controllers) {
            controller.processMouseMoved();
        }
        if (StageManager.getInstance().getDefaultStage() != StageManager.getInstance().getActiveStage()) {
            for (Controller controller : StageManager.getInstance().getDefaultStage().controllers) {
                controller.processMouseMoved();
            }
        }
    }

    private void mouseWheel(MouseEvent me) {
        for (Controller controller : StageManager.getInstance().getActiveStage().controllers) {
            controller.processMouseWheel(me);
        }
        if (StageManager.getInstance().getDefaultStage() != StageManager.getInstance().getActiveStage()) {
            for (Controller controller : StageManager.getInstance().getDefaultStage().controllers) {
                controller.processMouseWheel(me);
            }
        }
    }

    /**
     * This is a 'transfer' method - it 'redirects' the PApplet.keyPressed event to KTGUI components (controllers)
     */
    private void keyPressed() {
        for (Controller controller : StageManager.getInstance().getActiveStage().controllers) {
            controller.processKeyPressed();
        }
        if (StageManager.getInstance().getDefaultStage() != StageManager.getInstance().getActiveStage()) {
            for (Controller controller : StageManager.getInstance().getDefaultStage().controllers) {
                controller.processKeyPressed();
            }
        }
    }

    /**
     * This is a 'transfer' method - it 'redirects' the PApplet.keyReleased event to KTGUI components (controllers)
     */
    private void keyReleased() {
        for (Controller controller : StageManager.getInstance().getActiveStage().controllers) {
            controller.processKeyReleased();
        }
        if (StageManager.getInstance().getDefaultStage() != StageManager.getInstance().getActiveStage()) {
            for (Controller controller : StageManager.getInstance().getDefaultStage().controllers) {
                controller.processKeyReleased();
            }
        }
    }

}
