declare namespace fisica {
    interface FContactListener {
        contactStarted(contact: fisica.FContact): any;
        contactPersisted(contact: fisica.FContact): any;
        contactEnded(contact: fisica.FContact): any;
        contactResult(result: fisica.FContactResult): any;
    }
}
declare namespace fisica {
    /**
     * Provides default do-nothing implementations of all {@link FContactListener}
     * methods.
     * @class
     */
    class FContactAdapter implements fisica.FContactListener {
        /**
         *
         * @param {fisica.FContact} contact
         */
        contactStarted(contact: fisica.FContact): void;
        /**
         *
         * @param {fisica.FContact} contact
         */
        contactPersisted(contact: fisica.FContact): void;
        /**
         *
         * @param {fisica.FContact} contact
         */
        contactEnded(contact: fisica.FContact): void;
        /**
         *
         * @param {fisica.FContactResult} result
         */
        contactResult(result: fisica.FContactResult): void;
        constructor();
    }
}
declare namespace fisica {
    /**
     * This is a class from which all drawable objects of a world inherit.
     *
     * It contains most helping methods that handles stroke and fill colors and styles, as well as image attachments, etc.
     * @class
     */
    abstract class FDrawable {
        m_drawable: boolean;
        m_fill: boolean;
        m_fillColor: number;
        m_stroke: boolean;
        m_strokeColor: number;
        m_strokeWeight: number;
        m_image: PImage;
        m_imageAlpha: number;
        m_mask: PImage;
        updateStyle(other: FDrawable): void;
        appletStroke(applet: PGraphics, argb: number): void;
        appletFill(applet: PGraphics, argb: number): void;
        appletFillStroke(applet: PGraphics): void;
        drawImage(applet: PGraphics): void;
        /**
         * This method is called when calling {@code world.draw()}.
         * This method may be overriden to allow custom drawing of the object.
         *
         * @param {PGraphics} graphics  the graphics onto which the object must be drawn.
         */
        draw(graphics: PGraphics): void;
        /**
         * This method is called when calling {@code world.drawDebug()}.
         * This method may be overriden to allow custom debug drawing of the object.
         *
         * @param {PGraphics} graphics  the graphics onto which the object must be drawn.
         */
        drawDebug(graphics: PGraphics): void;
        /**
         * Attach an image to the object.
         * This method allows to draw an image onto the screen instead of calling the {@link #draw(PApplet)} method.
         *
         * @param {PImage} img  the PImage to attach to the object.
         */
        attachImage(img: PImage): void;
        /**
         * Dettach any image that has been attached to the object.
         *
         * @see #attachImage(PImage)
         */
        dettachImage(): void;
        /**
         * Get the opacity with which to draw the attached image.
         *
         * @return {number} the opacity, a value from 0.0 to 1.0 with which to draw the attached image
         * @see #attachImage(PImage)
         * @see #setImageAlpha(float)
         */
        getImageAlpha(): number;
        /**
         * Set the opacity with which to draw the attached image.
         *
         * @param {number} alpha   the opacity, a value from 0.0 to 1.0 with which to draw the attached image
         * @see #attachImage(PImage)
         * @see #getImageAlpha()
         */
        setImageAlpha(alpha: number): void;
        /**
         * Set whether the object must be drawn or not.
         *
         * @param {boolean} drawable  if {@code true} the object will be drawn, else it will not
         * @see #isDrawable()
         */
        setDrawable(drawable: boolean): void;
        /**
         * Get whether the object must be drawn or not.
         *
         * @return {boolean} drawable  if {@code true} the object will be drawn, else it will not
         * @see #setDrawable(boolean)
         */
        isDrawable(): boolean;
        /**
         * Returns the fill color of the object.
         *
         * @return {number} a Processing color type. e.g.  {@code myBody.setFillColor(color(20,100,30,90));}
         * @see #setNoFill()
         * @see #setFill(float)
         * @see #setFill(float,float)
         * @see #setFill(float,float,float)
         * @see #setFill(float,float,float,float)
         */
        getFillColor(): number;
        /**
         * Set the fill color of the object.  This method must be used in conjunction with Processing's color().  In most cases users will find it more convenient to use the versions of {@link #setFill(float)}, {@link #setFill(float,float)}, {@link #setFill(float,float,float)} or {@link #setFill(float,float,float,float)}
         *
         * @param {number} col  a Processing color type. e.g.  {@code myBody.setFillColor(color(20,100,30,90));}
         * @see #setNoFill()
         * @see #setFill(float)
         * @see #setFill(float,float)
         * @see #setFill(float,float,float)
         * @see #setFill(float,float,float,float)
         */
        setFillColor(col: number): void;
        /**
         * Set that the object must  be drawn without fill.
         *
         * @see #setFill(float)
         * @see #setFill(float,float)
         * @see #setFill(float,float,float)
         * @see #setFill(float,float,float,float)
         */
        setNoFill(): void;
        setFill$float(g: number): void;
        setFill$float$float(g: number, a: number): void;
        setFill$float$float$float(r: number, g: number, b: number): void;
        setFill$float$float$float$float(r: number, g: number, b: number, a: number): void;
        /**
         * Set the fill color of the object.
         *
         * @param {number} r   red value
         * @param {number} g   green value
         * @param {number} b   blue value
         * @param {number} a   alpha (opacity) value
         * @see #setFill(float)
         * @see #setFill(float,float)
         * @see #setFill(float,float,float)
         * @see #setFill(float,float,float,float)
         */
        setFill(r?: any, g?: any, b?: any, a?: any): any;
        /**
         * Set the stroke color of the object.  This method must be used in conjunction with Processing's color().  In most cases users will find it more convenient to use the versions of {@link #setStroke(float)}, {@link #setStroke(float,float)}, {@link #setStroke(float,float,float)} or {@link #setStroke(float,float,float,float)}
         *
         * @param {number} col  a Processing color type. e.g.  {@code myBody.setStrokeColor(color(20,100,30,90));}
         * @see #setNoStroke()
         * @see #setStroke(float)
         * @see #setStroke(float,float)
         * @see #setStroke(float,float,float)
         * @see #setStroke(float,float,float,float)
         */
        setStrokeColor(col: number): void;
        /**
         * Set that the object must  be drawn without stroke.
         *
         * @see #setStroke(float)
         * @see #setStroke(float,float)
         * @see #setStroke(float,float,float)
         * @see #setStroke(float,float,float,float)
         */
        setNoStroke(): void;
        setStroke$float(g: number): void;
        setStroke$float$float(g: number, a: number): void;
        setStroke$float$float$float(r: number, g: number, b: number): void;
        setStroke$float$float$float$float(r: number, g: number, b: number, a: number): void;
        /**
         * Set the stroke color of the object.
         *
         * @param {number} r   red value
         * @param {number} g   green value
         * @param {number} b   blue value
         * @param {number} a   alpha (opacity) value
         * @see #setStroke(float)
         * @see #setStroke(float,float)
         * @see #setStroke(float,float,float)
         * @see #setStroke(float,float,float,float)
         */
        setStroke(r?: any, g?: any, b?: any, a?: any): any;
        /**
         * Set the stroke weight of the object.
         *
         * @param {number} weight   weight value in pixels
         * @see #setStroke(float)
         * @see #setStroke(float,float)
         * @see #setStroke(float,float,float)
         * @see #setStroke(float,float,float,float)
         */
        setStrokeWeight(weight: number): void;
    }
}
declare namespace fisica {
    /**
     * Represents the result of a casted ray.
     *
     *
     * <pre>
     * {@code
     * FWorld world;
     *
     * void setup() {
     * Fisica.init(this);
     *
     * world = new FWorld();
     * world.setEdges();
     *
     * // Create and add bodies to the world here
     * // ...
     * }
     *
     * void draw() {
     * world.step();
     * world.draw();
     *
     * FRaycastResult result = null;
     * FBody b = world.raycastOne(width/2, height/2, mouseX, mouseY, result, true);
     * }
     *
     * }
     * </pre>
     *
     *
     * @usage Contacts
     * @see FContact
     * @class
     */
    class FRaycastResult {
        m_lambda: number;
        m_normal: org.jbox2d.common.Vec2;
        m_x1: number;
        m_x2: number;
        m_y1: number;
        m_y2: number;
        set(x1: number, y1: number, x2: number, y2: number, raycastResult: org.jbox2d.common.RaycastResult): FRaycastResult;
        /**
         * Returns the lambda of the raycast result.
         *
         * @return {number} the lambda of the raycast result
         */
        getLambda(): number;
        /**
         * Returns the horizontal component of the ray cast contact normal.
         *
         * @return {number} the horizontal component of the ray cast contact normal
         * @see #getNormalY
         */
        getNormalX(): number;
        /**
         * Returns the vertical component of the ray cast contact normal.
         *
         * @return {number} the vertical component of the ray cast contact normal
         * @see #getNormalX
         */
        getNormalY(): number;
        /**
         * Returns the horizontal component of the ray cast contact normal.
         *
         * @return {number} the horizontal component of the ray cast contact normal
         * @see #getY
         * @see #getNormalX
         * @see #getNormalY
         */
        getX(): number;
        /**
         * Returns the vertical component of the contact ray cast normal.
         *
         * @return {number} the vertical component of the contact ray cast normal
         * @see #getX
         * @see #getNormalX
         * @see #getNormalY
         */
        getY(): number;
        constructor();
    }
}
declare namespace fisica {
    /**
     * Represents the library Fisica.  It is mainly used to initialize it with the PApplet:
     * <pre>
     * {@code
     * Fisica.init(this);
     * }
     * </pre>
     * It also allows to modify global properties of the simulation such as the scale of meters to pixels:
     * <pre>
     * {@code
     * Fisica.setScale(40);
     * }
     * </pre>
     * In the future it may contain helping methods to create the objects.
     * @class
     */
    class Fisica implements PConstants {
        static m_initialized: boolean;
        static m_parent: any;
        static m_viewport: Fisica.FViewport;
        static m_parentGraphics: PGraphics;
        static screenToWorld$org_jbox2d_common_Vec2(m_in: org.jbox2d.common.Vec2): org.jbox2d.common.Vec2;
        static screenToWorld$float$float(x: number, y: number): org.jbox2d.common.Vec2;
        static screenToWorld(x?: any, y?: any): any;
        static screenToWorld$float(a: number): number;
        static worldToScreen$org_jbox2d_common_Vec2(m_in: org.jbox2d.common.Vec2): org.jbox2d.common.Vec2;
        static worldToScreen$float$float(x: number, y: number): org.jbox2d.common.Vec2;
        static worldToScreen(x?: any, y?: any): any;
        static worldToScreen$float(a: number): number;
        static initialized(): boolean;
        static parent(): any;
        static parentGraphics(): PGraphics;
        /**
         * Initialize the library.  Must be called before any use of the library.  Must be called by passing the PApplet.  e.g. {@code Fisica.init(this)}
         *
         * @param {*} applet  The applet on which to use the library.  This library can only be used with one applet
         */
        static init(applet: any): void;
        /**
         * Set the scale from screen units to world units.  By setting the scale to 20 we are stating that 20 pixels is equivalent to 1 meter in the simulated world.
         *
         * @param {number} scale the number of pixels that are equivalent to 1 meter in the simulated world.
         */
        static setScale(scale: number): void;
        constructor();
    }
    namespace Fisica {
        class FViewport {
            m_scale: number;
            constructor();
            setScaleTransform(a: number): void;
            getScreenToWorld$float(a: number): number;
            getScreenToWorld$float$float(x: number, y: number): org.jbox2d.common.Vec2;
            getScreenToWorld(x?: any, y?: any): any;
            getScreenToWorld$org_jbox2d_common_Vec2(p: org.jbox2d.common.Vec2): org.jbox2d.common.Vec2;
            getWorldToScreen$float(a: number): number;
            getWorldToScreen$float$float(x: number, y: number): org.jbox2d.common.Vec2;
            getWorldToScreen(x?: any, y?: any): any;
            getWorldToScreen$org_jbox2d_common_Vec2(p: org.jbox2d.common.Vec2): org.jbox2d.common.Vec2;
        }
        /**
         * Exception thrown when the library has not been initialized.  The method {@link Fisica#init(PApplet)} must be called before any use of the library.
         *
         * @param applet  The applet on which to use the library.  This library can only be used with one applet
         * @extends Error
         * @class
         */
        class LibraryNotInitializedException {
            static __fisica_Fisica_LibraryNotInitializedException_serialVersionUID: number;
            constructor();
        }
    }
}
declare namespace fisica {
    class FContactID {
        m_id: org.jbox2d.collision.ContactID;
        m_b1: fisica.FBody;
        m_b2: fisica.FBody;
        static HASH_PRIME: number;
        constructor(id?: any, b1?: any, b2?: any);
        hashCode(): number;
        toString(): string;
        equals(obj: any): boolean;
    }
}
declare namespace fisica {
    /**
     * Represents a contact between two bodies.  Objects of this type are not created by the users.  Contacts are passed to the user when they implement the {@code contactStarted(FContact){ }}, {@code contactPersisted(FContact){ }} and {@code contactEnded(FContact){ }} methods in the applet:
     *
     *
     * <pre>
     * {@code
     * FWorld world;
     *
     * void setup() {
     * Fisica.init(this);
     *
     * world = new FWorld();
     * world.setEdges();
     *
     * // Create and add bodies to the world here
     * // ...
     * }
     *
     * void draw() {
     * world.step();
     * world.draw();
     * }
     *
     * void contactStarted(FContact contact) {
     * // Draw in green an ellipse where the contact took place
     * fill(0, 170, 0);
     * ellipse(contact.getX(), contact.getY(), 20, 20);
     * }
     *
     * void contactPersisted(FContact contact) {
     * // Draw in blue an ellipse where the contact took place
     * fill(0, 0, 170);
     * ellipse(contact.getX(), contact.getY(), 10, 10);
     * }
     *
     * void contactStarted(FContact contact) {
     * // Draw in red an ellipse where the contact took place
     * fill(170, 0, 0);
     * ellipse(contact.getX(), result.getY(), 20, 20);
     * }
     *
     * }
     * </pre>
     *
     * To know if the contact is the beggining, the continuation or the end of a contact it is better to use the other method {@code contactResult(FContactResult){ }}.
     *
     * @usage Contacts
     * @see FContactResult
     * @class
     */
    class FContact {
        m_body1: fisica.FBody;
        m_body2: fisica.FBody;
        m_position: org.jbox2d.common.Vec2;
        m_velocity: org.jbox2d.common.Vec2;
        m_normal: org.jbox2d.common.Vec2;
        m_separation: number;
        m_friction: number;
        m_restitution: number;
        m_id: fisica.FContactID;
        constructor(contactPoint: org.jbox2d.dynamics.contacts.ContactPoint);
        /**
         * Returns the first body involved in the contact.
         * @return {fisica.FBody} first of the bodies involved in the contact
         */
        getBody1(): fisica.FBody;
        /**
         * Returns the second body involved in the contact.
         * @return {fisica.FBody} second of the bodies involved in the contact
         */
        getBody2(): fisica.FBody;
        /**
         * Returns the horizontal position of the contact point.
         *
         * @return {number} the horizontal position of the contact point in pixels
         * @see #getY
         */
        getX(): number;
        /**
         * Returns the vertical position of the contact point.
         *
         * @return {number} the vertical position of the contact point in pixels
         * @see #getX
         */
        getY(): number;
        /**
         * Returns the horizontal component of the contact velocity.
         *
         * @return {number} the horizontal component of the contact velocity
         * @see #getVelocityY
         */
        getVelocityX(): number;
        /**
         * Returns the vertical component of the contact velocity.
         *
         * @return {number} the vertical component of the contact velocity
         * @see #getVelocityX
         */
        getVelocityY(): number;
        /**
         * Returns the horizontal component of the contact normal.
         *
         * @return {number} the horizontal component of the contact normal
         * @see #getNormalY
         */
        getNormalX(): number;
        /**
         * Returns the vertical component of the contact normal.
         *
         * @return {number} the vertical component of the contact normal
         * @see #getNormalX
         */
        getNormalY(): number;
        /**
         * Get the separation between the bodies.
         *
         * @return {number} a positive value means that the bodies have space between them, negative values means that the bodies have penetrated each other
         */
        getSeparation(): number;
        /**
         * Get the friction coefficient of the contact.  The friction determines the ratio of the reaction force tangent to a contact that the two bodies will recieve.  Basically it can be seen as a coefficient that will control how the bodies get slown down when they slide against each other.  This value depends on the friction coefficients of the two bodies involved in the contact.
         *
         * @return {number} a positive value.  A value of 0 means no friction and thus the body will not be slown down if no other forces are applied
         */
        getFriction(): number;
        /**
         * Get the restitution coefficient of the contact.  The restitution determines the ratio of the reaction force normal to a contact that the two bodies will recieve.  Basically it can be seen as a coefficient that will control the strength with which the bodies bounce back from the collision.  This value depends on the resititution coefficients of the two bodies involved in the contact.
         *
         * @return {number} a positive value.  A value of 0 means no bounce after a collision, and a value of 1 means bounce with it's full speed from a collision
         */
        getRestitution(): number;
        /**
         * Get the identifier of the contact.  This value is useful in order to uniquely identify a contact.  A new contact ID is created whenever to bodies enter into contact at a given point.  If the bodies slide against each other the contact ID is maintained even if the point of contact is modified due to the slide.  As soon as the two bodies separate the contact is considered ended.
         *
         * @return {fisica.FContactID} a unique identifier representing the contact
         */
        getId(): fisica.FContactID;
        contains$java_lang_String$java_lang_String(n1: string, n2: string): boolean;
        /**
         * Returns true if the contact contains the two bodies.  If one of the bodies does not have a name this function returns false.
         *
         * @param {string} n1 the name of one of the bodies
         * @param {string} n2 the name of another one of the bodies
         * @return {boolean} true if the contact bodies have the names given by the parameters
         * @see FBody#setName
         */
        contains(n1?: any, n2?: any): any;
        contains$fisica_FBody$fisica_FBody(n1: fisica.FBody, n2: fisica.FBody): boolean;
        contains$java_lang_String(n1: string): boolean;
        contains$fisica_FBody(n1: fisica.FBody): boolean;
    }
}
declare namespace fisica {
    /**
     * Represents the result of the contact between two bodies.  Objects of this type are not created by the users.  Contact results are passed to the user when they implement the {@code contactResult(FContactResult){ }} method in the applet:
     *
     *
     * <pre>
     * {@code
     * FWorld world;
     *
     * void setup() {
     * Fisica.init(this);
     *
     * world = new FWorld();
     * world.setEdges();
     *
     * // Create and add bodies to the world here
     * // ...
     * }
     *
     * void draw() {
     * world.step();
     * world.draw();
     * }
     *
     * void contactResult(FContactResult result) {
     * // Draw an ellipse where the contact took place and as big as the normal impulse of the contact
     * ellipse(result.getX(), result.getY(), result.getNormalImpulse(), result.getNormalImpulse());
     *
     * // Trigger your sound here
     * // ...
     * }
     * }
     * </pre>
     *
     * To know if the contact is the beggining, the continuation or the end of a contact it is better to use the other methods {@code contactStarted(FContact){ }}, {@code contactPersisted(FContact){ }} and {@code contactEnded(FContact){ }}.
     *
     * @usage Contacts
     * @see FContact
     * @class
     */
    class FContactResult {
        m_position: org.jbox2d.common.Vec2;
        m_normal: org.jbox2d.common.Vec2;
        m_body1: fisica.FBody;
        m_body2: fisica.FBody;
        m_normalImpulse: number;
        m_tangentImpulse: number;
        m_id: fisica.FContactID;
        constructor(contactResult: org.jbox2d.dynamics.contacts.ContactResult);
        /**
         * Returns the first body involved in the contact.
         * @return {fisica.FBody} first of the bodies involved in the contact
         */
        getBody1(): fisica.FBody;
        /**
         * Returns the second body involved in the contact.
         * @return {fisica.FBody} second of the bodies involved in the contact
         */
        getBody2(): fisica.FBody;
        /**
         * Returns the horizontal position of the contact point.
         *
         * @return {number} the horizontal position of the contact point in pixels
         * @see #getY
         */
        getX(): number;
        /**
         * Returns the vertical position of the contact point.
         *
         * @return {number} the vertical position of the contact point in pixels
         * @see #getX
         */
        getY(): number;
        /**
         * Returns the horizontal component of the contact normal.
         *
         * @return {number} the horizontal component of the contact normal
         * @see #getNormalY
         */
        getNormalX(): number;
        /**
         * Returns the vertical component of the contact normal.
         *
         * @return {number} the vertical component of the contact normal
         * @see #getNormalX
         */
        getNormalY(): number;
        /**
         * Returns the normal component of the impulse of the contact.  This gives an idea of the strength of the collision that took place.  This represents the impulse necessary to avoid penetration of the bodies involved in the collision.  The impluse is simply the force multiplied by the timestep.  The result is returned in impulse units (kg * pixels / s).
         *
         * @return {number} the normal component of the contact's impulse
         * @see #getTangentImpulse
         */
        getNormalImpulse(): number;
        /**
         * Returns the tangential component of the impulse of the contact.  This gives an idea of the strength of the friction between the bodies that took place.  The impluse is simply the force multiplied by the timestep.  The result is returned in impulse units (kg * pixels / s).
         *
         * @return {number} the tangent component of the contact's impulse
         * @see #getNormalImpulse
         */
        getTangentImpulse(): number;
        /**
         * Get the identifier of the contact.  This value is useful in order to uniquely identify a contact.  A new contact ID is created whenever to bodies enter into contact at a given point.  If the bodies slide against each other the contact ID is maintained even if the point of contact is modified due to the slide.  As soon as the two bodies separate the contact is considered ended.
         *
         * @return {fisica.FContactID} a unique identifier representing the contact
         */
        getId(): fisica.FContactID;
    }
}
declare namespace fisica {
    /**
     * Constructs the world where all the bodies live in. We usually want to build
     * the world larger than the actual screen, because when bodies exit the world
     * they will appear stuck since they do not get update anymore. By default the
     * world's width and height are three times larger than those of the Processing
     * canvas.
     *
     * {@code
     * FWorld world;
     *
     * void setup() {
     * size(200, 200);
     *
     * Fisica.init(this); world = new FWorld(-width, -height, 2*width, 2*height); }
     * }
     *
     * @usage World
     * @param {number} topLeftX
     * the horizontal coordinate of the top left corner of the world
     * @param {number} topLeftY
     * the vertical coordinate of the top left corner of the world
     * @param {number} bottomRightX
     * the horizontal coordinate of the bottom right corner of the world
     * @param {number} bottomRightY
     * the vertical coordinate of the bottom right corner of the world
     * @see FBody
     * @class
     * @extends org.jbox2d.dynamics.World
     */
    class FWorld extends org.jbox2d.dynamics.World {
        /**
         * The left edge of the world. For this edge to exist, the edges must have been
         * created by calling {@link #setEdges()}.
         *
         * @see #left
         * @see #right
         * @see #bottom
         * @see #top
         */
        left: fisica.FBox;
        /**
         * The right edge of the world. For this edge to exist, the edges must have been
         * created by calling {@link #setEdges()}.
         *
         * @see #left
         * @see #right
         * @see #bottom
         * @see #top
         */
        right: fisica.FBox;
        /**
         * The top edge of the world. For this edge to exist, the edges must have been
         * created by calling {@link #setEdges()}.
         *
         * @see #left
         * @see #right
         * @see #bottom
         * @see #top
         */
        top: fisica.FBox;
        /**
         * The bottom edge of the world. For this edge to exist, the edges must have
         * been created by calling {@link #setEdges()}.
         *
         * @see #left
         * @see #right
         * @see #bottom
         * @see #top
         */
        bottom: fisica.FBox;
        m_topLeftX: number;
        m_topLeftY: number;
        m_bottomRightX: number;
        m_bottomRightY: number;
        m_edgesFriction: number;
        m_edgesRestitution: number;
        m_grabbable: boolean;
        m_grabPositionX: number;
        m_grabPositionY: number;
        m_mouseButton: number;
        m_contacts: any;
        m_contactResults: Array<any>;
        m_actions: Array<any>;
        m_fbodies: Array<any>;
        m_mouseJoint: fisica.FMouseJoint;
        m_small: org.jbox2d.common.Vec2;
        m_aabb: org.jbox2d.collision.AABB;
        addBody(body: fisica.FBody): void;
        removeBody(body: fisica.FBody): void;
        addJoint(joint: fisica.FJoint): void;
        removeJoint(joint: fisica.FJoint): void;
        m_contactListener: FWorld.ConcreteContactListener;
        /**
         * A PApplet can either provide an {@link FContactListener} for receiving
         * contact events, or can implement the contact event methods itself without
         * explicitly implementing {@link FContactListener}.
         */
        m_clientContactListener: fisica.FContactListener;
        m_contactStartedMethod: (p1: fisica.FContact) => void;
        m_contactPersistedMethod: (p1: fisica.FContact) => void;
        m_contactEndedMethod: (p1: fisica.FContact) => void;
        m_contactResultMethod: (p1: fisica.FContactResult) => void;
        setContactListener$fisica_FContactListener(listener: fisica.FContactListener): void;
        setContactListener(listener?: any): any;
        setContactListener$org_jbox2d_dynamics_ContactListener(listener: org.jbox2d.dynamics.ContactListener): void;
        grabBody(x: number, y: number): void;
        dragBody(x: number, y: number): void;
        releaseBody(): void;
        /**
         * This is an internal method to handle mouse interaction and should not be
         * used.
         *
         * @internal
         * @exclude
         * @param {ProcessingMouseEvent} event
         */
        mouseEvent(event: ProcessingMouseEvent): void;
        constructor(topLeftX?: any, topLeftY?: any, bottomRightX?: any, bottomRightY?: any);
        /**
         * Returns the mouse joint that is used for interaction with the bodies in the
         * world.
         *
         * @return {fisica.FMouseJoint} the mouse joint used for grabbing bodies
         */
        getMouseJoint(): fisica.FMouseJoint;
        /**
         * Controls whether the bodies in the world can be grabbed by the mouse or not.
         * By default the world bodies' are grabbable and draggable.
         *
         * {@code
         * world.setGrabbable(false);
         * }
         *
         * @usage World
         * @param {boolean} value
         * if true the bodies that live in this world can be grabbed and
         * dragged using the mouse
         * @see FBody
         */
        setGrabbable(value: boolean): void;
        processActions(): void;
        draw$def_processing_core_PGraphics(graphics: PGraphics): void;
        /**
         * Draws all the bodies in the world. This method is often called in the draw
         * method of the applet.
         *
         * @param {PGraphics} graphics
         * graphics to which to draw the world. Useful when trying to draw
         * the world on other buffers, such as when using createGraphics
         * @see FBody
         */
        draw(graphics?: any): any;
        drawDebug$def_processing_core_PGraphics(graphics: PGraphics): void;
        /**
         * Draws the debug version of all the bodies in the world. This method is often
         * called in the draw method of the applet.
         *
         * @param {PGraphics} graphics
         * graphics to which to draw the world. Useful when trying to draw
         * the world on other buffers, such as when using createGraphics
         * @see FBody
         */
        drawDebug(graphics?: any): any;
        draw$(): void;
        drawDebug$(): void;
        add$fisica_FBody(body: fisica.FBody): void;
        /**
         * Add a body to the world.
         *
         * @param {fisica.FBody} body
         * body to be added to the world
         * @see #remove(FBody)
         */
        add(body?: any): any;
        remove$fisica_FBody(body: fisica.FBody): void;
        /**
         * Remove a body from the world.
         *
         * @param {fisica.FBody} body
         * body to be removed from the world
         * @see #add(FBody)
         */
        remove(body?: any): any;
        add$fisica_FJoint(joint: fisica.FJoint): void;
        remove$fisica_FJoint(joint: fisica.FJoint): void;
        /**
         * Clear all bodies and joints from the world. NOT IMPLEMENTED YET.
         */
        clear(): void;
        setEdges$float$float$float$float$int(topLeftX: number, topLeftY: number, bottomRightX: number, bottomRightY: number, color: number): void;
        /**
         * Add edges of given dimensions to the world. This will create the bodies for
         * {@link #left}, {@link #right}, {@link #bottom} and {@link #top}.
         *
         * @param {number} topLeftX
         * the horizontal coordinate of the top left corner of the edges
         * @param {number} topLeftY
         * the vertical coordinate of the top left corner of the edges
         * @param {number} bottomRightX
         * the horizontal coordinate of the bottom right corner of the edges
         * @param {number} bottomRightY
         * the vertical coordinate of the bottom right corner of the edges
         * @param {number} color
         * the color of the edges. This color must be passed using
         * Processing's color() function
         */
        setEdges(topLeftX?: any, topLeftY?: any, bottomRightX?: any, bottomRightY?: any, color?: any): any;
        setEdges$float$float$float$float(topLeftX: number, topLeftY: number, bottomRightX: number, bottomRightY: number): void;
        setEdges$def_processing_core_PApplet$int(applet: any, color: number): void;
        setEdges$int(color: number): void;
        setEdges$(): void;
        /**
         * Set the friction of all the edges.
         *
         * @param {number} friction
         * the friction of the edges
         */
        setEdgesFriction(friction: number): void;
        /**
         * Set the restitution of all the edges.
         *
         * @param {number} restitution
         * the restitution of the edges
         */
        setEdgesRestitution(restitution: number): void;
        setGravity$float$float(gx: number, gy: number): void;
        /**
         * Set the gravity of the world. Use {@code world.setGravity(0,0);} to have a
         * world without gravity.
         *
         * @param {number} gx
         * the horizontal component of the gravity
         * @param {number} gy
         * the vertical component of the gravity
         */
        setGravity(gx?: any, gy?: any): any;
        setGravity$org_jbox2d_common_Vec2(vector: org.jbox2d.common.Vec2): void;
        step$(): void;
        step$float(dt: number): void;
        step$float$int(dt: number, iterationCount: number): void;
        /**
         * Advance the world simulation of given time, with a given number of
         * iterations. The larger the number of iterations, the more computationally
         * expensive and precise it will be. The default is 10 iterations.
         *
         * @param {number} dt
         * the time to advance the world simulation
         * @param {number} iterationCount
         * the number of iterations for the world simulation step
         */
        step(dt?: any, iterationCount?: any): any;
        getBody$float$float(x: number, y: number): fisica.FBody;
        getBody$float$float$boolean(x: number, y: number, getStatic: boolean): fisica.FBody;
        /**
         * Returns the first object found at the given position.
         *
         * @param {number} x
         * the horizontal component of the position
         * @param {number} y
         * the vertical component of the position
         * @param {boolean} getStatic
         * if {@code true} it will also get static objects that can be found
         * at that position
         * @return {fisica.FBody} the body found at the given position
         */
        getBody(x?: any, y?: any, getStatic?: any): any;
        getBodies$(): Array<any>;
        getBodies$float$float(x: number, y: number): Array<any>;
        getBodies$float$float$boolean(x: number, y: number, getStatic: boolean): Array<any>;
        getBodies$float$float$boolean$int(x: number, y: number, getStatic: boolean, count: number): Array<any>;
        /**
         * Returns a list with all the bodies found at the given position.
         *
         * @param {number} x
         * the horizontal component of the position
         * @param {number} y
         * the vertical component of the position
         * @param {boolean} getStatic
         * if {@code true} it will also get static objects that can be found
         * at that position
         * @param {number} count
         * the maximum number of bodies to be retrieved
         * @return {Array} an ArrayList (of FBody) of all bodies found at the given position
         */
        getBodies(x?: any, y?: any, getStatic?: any, count?: any): any;
        raycast$float$float$float$float$fisica_FBody_A$int$boolean(x1: number, y1: number, x2: number, y2: number, bodies: fisica.FBody[], maxCount: number, solidShapes: boolean): number;
        raycast(x1?: any, y1?: any, x2?: any, y2?: any, bodies?: any, maxCount?: any, solidShapes?: any): any;
        raycast$org_jbox2d_collision_Segment$org_jbox2d_collision_shapes_Shape_A$int$boolean$java_lang_Object(segment: org.jbox2d.collision.Segment, shapes: org.jbox2d.collision.shapes.Shape[], maxCount: number, solidShapes: boolean, userData: any): number;
        raycastOne$org_jbox2d_collision_Segment$org_jbox2d_common_RaycastResult$boolean$java_lang_Object(segment: org.jbox2d.collision.Segment, result: org.jbox2d.common.RaycastResult, solidShapes: boolean, userData: any): org.jbox2d.collision.shapes.Shape;
        raycastOne$float$float$float$float$fisica_FRaycastResult$boolean(x1: number, y1: number, x2: number, y2: number, result: fisica.FRaycastResult, solidShapes: boolean): fisica.FBody;
        raycastOne(x1?: any, y1?: any, x2?: any, y2?: any, result?: any, solidShapes?: any): any;
    }
    namespace FWorld {
        /**
         * Forward the contact events to the contactStarted(ContactPoint point),
         * contactPersisted(ContactPoint point) and contactStopped(ContactPoint point)
         * which may be implemented in the sketch.
         * @class
         */
        class ConcreteContactListener implements org.jbox2d.dynamics.ContactListener {
            __parent: any;
            add(point: org.jbox2d.dynamics.contacts.ContactPoint): void;
            persist(point: org.jbox2d.dynamics.contacts.ContactPoint): void;
            remove(point: org.jbox2d.dynamics.contacts.ContactPoint): void;
            m_world: fisica.FWorld;
            result(point: org.jbox2d.dynamics.contacts.ContactResult): void;
            constructor(__parent: any);
        }
    }
}
declare namespace fisica {
    abstract class FWorldAction {
        abstract apply(world: fisica.FWorld): any;
    }
}
declare namespace fisica {
    /**
     * Represents a body in the world.
     *
     * A body is an object which may collide and react to forces in the world.  The bodies have many properties such as density, velocity, position, etc... with which we can control their behavior.  This class cannot be be instantiated, instead use one of the derived classes.
     * @class
     * @extends fisica.FDrawable
     */
    abstract class FBody extends fisica.FDrawable {
        m_density: number;
        m_restitution: number;
        m_friction: number;
        m_bullet: boolean;
        m_sensor: boolean;
        m_static: boolean;
        m_linearDamping: number;
        m_angularDamping: number;
        m_rotatable: boolean;
        m_allowSleep: boolean;
        m_isSleeping: boolean;
        m_groupIndex: number;
        m_filterBits: number;
        m_categoryBits: number;
        m_linearVelocity: org.jbox2d.common.Vec2;
        m_angularVelocity: number;
        m_force: org.jbox2d.common.Vec2;
        m_torque: number;
        m_position: org.jbox2d.common.Vec2;
        m_angle: number;
        m_name: string;
        m_body: org.jbox2d.dynamics.Body;
        m_world: fisica.FWorld;
        m_parent: FBody;
        m_grabbable: boolean;
        m_shape: org.jbox2d.collision.shapes.Shape;
        processBody(bd: org.jbox2d.dynamics.Body, sd: org.jbox2d.collision.shapes.ShapeDef): void;
        /**
         * WARNING: This method is internal only and may change someday.  If you are using this method please contact the developer since there should be a better way or we may add something to the library.
         *
         * @return {org.jbox2d.dynamics.Body} the internal JBox2D body
         */
        getBox2dBody(): org.jbox2d.dynamics.Body;
        /**
         * Get the group to which this body belongs.  Groups allow to select the bodies that may collide together or with others.  If the group index is negative then they will not collide with each other but they will collide with all the bodies of the other groups.
         *
         * @return {number} the index of the group
         */
        getGroupIndex(): number;
        addToWorld(world: fisica.FWorld): void;
        setState(b: FBody): void;
        setStateFromWorld(): void;
        recreateInWorld(): void;
        removeFromWorld(): void;
        getShapeDef(): org.jbox2d.collision.shapes.ShapeDef;
        getTransformedShapeDef(): org.jbox2d.collision.shapes.ShapeDef;
        getProcessedShapeDef(): org.jbox2d.collision.shapes.ShapeDef;
        getShapeDefs(): Array<any>;
        getBodies(): Array<any>;
        processShapeDef(sd: org.jbox2d.collision.shapes.ShapeDef): org.jbox2d.collision.shapes.ShapeDef;
        preDraw(applet: PGraphics): void;
        postDraw(applet: PGraphics): void;
        preDrawDebug(applet: PGraphics): void;
        postDrawDebug(applet: PGraphics): void;
        getAABB(): org.jbox2d.collision.AABB;
        getBB(): org.jbox2d.collision.AABB;
        applyMatrix(applet: PGraphics): void;
        /**
         * Control the group to which this body belongs.  Groups allow to select the bodies that may collide together or with others.  If the group index is negative then they will not collide with each other but they will collide with all the bodies of the other groups.
         *
         * @param {number} index  the index of the group
         */
        setGroupIndex(index: number): void;
        setFilterBits(mask: number): void;
        setCategoryBits(mask: number): void;
        getCategoryBits(): number;
        getFilterBits(): number;
        setParent(b: FBody): void;
        getParent(): FBody;
        /**
         * Control if this body can be grabbed by the mouse, when clicked on.  This property only has effect if the world is grabbable. If a body is grabbable, then it can be dragged around by the mouse.
         *
         * @see FWorld#setGrabbable(boolean)
         *
         * @param {boolean} value if {@code true} and the world it belongs to is grabbable, then the body is grabbable by the mouse
         */
        setGrabbable(value: boolean): void;
        /**
         * Set the force applied to the center of the body.
         *
         * @see #addForce(float,float)
         *
         * @param {number} fx the x coordinate of the force
         * @param {number} fy the y coordinate of the force
         */
        setForce(fx: number, fy: number): void;
        /**
         * Get the x coordinate of the force applied to the center of the body.
         *
         * @see #addForce(float,float)
         * @see #getForceY()
         *
         * @return {number} the x coordinate of the force
         */
        getForceX(): number;
        /**
         * Get the y coordinate of the force applied to the center of the body.
         *
         * @see #addForce(float,float)
         * @see #getForceX()
         *
         * @return {number} the y coordinate of the force
         */
        getForceY(): number;
        /**
         * Add a rotation force (a torque) to the body.
         *
         * @see #addForce(float,float)
         * @see #addForce(float,float,float,float)
         *
         * @param {number} torque the value of the torque
         */
        addTorque(torque: number): void;
        addForce$float$float(fx: number, fy: number): void;
        addImpulse$float$float(fx: number, fy: number): void;
        addForce$float$float$float$float(fx: number, fy: number, px: number, py: number): void;
        /**
         * Apply a force to a given point of the body.  If the force is not applied on the center of the body this force might induce a rotation change.  It would be as applying a force on the center of the body and a torque.
         *
         * @see #addTorque(float)
         * @see #addImpulse(float,float,float,float)
         *
         * @param {number} fx the x coordinate of the force
         * @param {number} fy the y coordinate of the force
         * @param {number} px the x position relative to the body's center, where to apply the force
         * @param {number} py the y position relative to the body's center, where to apply the force
         */
        addForce(fx?: any, fy?: any, px?: any, py?: any): any;
        addImpulse$float$float$float$float(fx: number, fy: number, px: number, py: number): void;
        /**
         * Apply an impulse to a given point of the body.  If the impulse is not applied on the center of the body this force might induce a rotation change.  It would be as applying a force on the center of the body and a torque.
         *
         * @see #addTorque(float)
         * @see #addForce(float,float,float,float)
         *
         * @param {number} fx the x coordinate of the force
         * @param {number} fy the y coordinate of the force
         * @param {number} px the x position relative to the body's center, where to apply the force
         * @param {number} py the y position relative to the body's center, where to apply the force
         */
        addImpulse(fx?: any, fy?: any, px?: any, py?: any): any;
        /**
         * Remove all the forces that are applied to the body.
         */
        resetForces(): void;
        /**
         * Returns the horizontal velocity of the body.
         *
         * @return {number} the horizontal velocity of the body in pixels per second
         */
        getVelocityX(): number;
        /**
         * Returns the vertical velocity of the body.
         *
         * @return {number} the vertical velocity of the body in pixels per second
         */
        getVelocityY(): number;
        /**
         * Set the velocity of the body.
         *
         * @param {number} vx  the horizontal velocity of the body in pixels per second
         * @param {number} vy  the vertical velocity of the body in pixels per second
         */
        setVelocity(vx: number, vy: number): void;
        /**
         * Adjust the velocity of the body.
         *
         * @param {number} dvx  the horizontal velocity to be added to the body in pixels per second
         * @param {number} dvy  the vertical velocity to be added to the body in pixels per second
         */
        adjustVelocity(dvx: number, dvy: number): void;
        /**
         * Returns the horizontal position of the body.
         *
         * @return {number} the horizontal position of the body in pixels
         * @see #getY
         * @see #setPosition(float,float)
         */
        getX(): number;
        /**
         * Returns the vertical position of the body.
         *
         * @return {number} the vertical position of the body in pixels
         * @see #getX
         * @see #setPosition(float,float)
         */
        getY(): number;
        setPosition$float$float(x: number, y: number): void;
        /**
         * Set the position of the body.
         *
         * @param {number} x  the horizontal position of the body in pixels
         * @param {number} y  the vertical position of the body in pixels
         */
        setPosition(x?: any, y?: any): any;
        setPosition$org_jbox2d_common_Vec2(position: org.jbox2d.common.Vec2): void;
        /**
         * Adjust the position of the body.
         *
         * @param {number} dx  the horizontal position to be added to the body in pixels
         * @param {number} dy  the vertical position to be added to the body in pixels
         */
        adjustPosition(dx: number, dy: number): void;
        /**
         * Returns the rotation of the body.
         *
         * @return {number} the rotation of the body in radians
         * @see #setRotation(float)
         */
        getRotation(): number;
        /**
         * Set the rotation of the body.
         *
         * @param {number} w  the rotation of the body in radians
         * @see #getRotation()
         */
        setRotation(w: number): void;
        /**
         * Adjust the rotation of the body.
         *
         * @param {number} dw  the rotation to be added to the body in radians
         * @see #getRotation()
         * @see #setRotation(float)
         */
        adjustRotation(dw: number): void;
        /**
         * Deprecated. Please use isSleeping().
         *
         * @return {boolean} true if the body is resting
         */
        isResting(): boolean;
        /**
         * Indicates whether the body is in a sleeping state.
         *
         * The sleeping state of a body is reached when it has not moved or has not received any forces nor collisions for some time.
         *
         * @return {boolean} true if the body is sleeping
         * @see #wakeUp()
         * @see #setAllowSleeping(boolean)
         */
        isSleeping(): boolean;
        /**
         * Wake up the body from a sleeping state.
         *
         * @see #isSleeping()
         * @see #setAllowSleeping(boolean)
         */
        wakeUp(): void;
        /**
         * Returns the rotation velocity of the body.
         *
         * @return {number} the rotation velocity of the body in radians per second
         * @see #setAngularVelocity(float)
         * @see #adjustAngularVelocity(float)
         */
        getAngularVelocity(): number;
        /**
         * Set the rotation velocity of the body.
         *
         * @param {number} w   the rotation velocity of the body in radians per second
         */
        setAngularVelocity(w: number): void;
        /**
         * Adjust the rotation velocity of the body.
         *
         * @param {number} dw   the rotation velocity to be added to the body in radians per second
         * @see #getAngularVelocity()
         * @see #setAngularVelocity(float)
         */
        adjustAngularVelocity(dw: number): void;
        /**
         * Set the damping of the rotation movement of the body.  The damping constantly reduces the rotation velocity of the body.
         *
         * @param {number} damping   the damping of the rotation movement of the body
         * @see #setDamping(float)
         */
        setAngularDamping(damping: number): void;
        /**
         * Set the damping of the translation movement of the body.  The damping constantly reduces the translation velocity of the body.
         *
         * @param {number} damping   the damping of the translation movement of the body
         * @see #setAngularDamping(float)
         */
        setDamping(damping: number): void;
        /**
         * Set the name of the body.
         *
         * @param {string} name   the name of the body
         */
        setName(name: string): void;
        /**
         * Get the name of the body.
         *
         * @return {string} name    the name of the body
         */
        getName(): string;
        /**
         * Set the density of the body.  The density will determine the total mass of the body and thus it's behavior with respect to collisions, bounces, inertia, joints,...  When the density is set, the mass of the body is recalculated automatically given it's area.
         *
         * Note that a density of 0.0 corresponds to a mass of 0.0 independently of the area and the body will be considered static.
         *
         * @param {number} density   the density of the body
         */
        setDensity(density: number): void;
        /**
         * Get the density of the body.  The density determines the total mass of the body and thus it's behavior with respect to collisions, bounces, inertia, joints,...
         *
         * Note that a density of 0.0 corresponds to a mass of 0.0 independently of the area and the body will be considered static.
         *
         * @return {number} density   the density of the body
         */
        getDensity(): number;
        updateMass(): void;
        /**
         * Set whether the body is a sensor.  Sensor bodies act as normal bodies in the sense that they notify about contacts, however they do not collide with other bodies (they act like ghost bodies).
         *
         * @param {boolean} value   if {@code true} the body will be a sensor.  It will not collide when enters contact with other bodies
         */
        setSensor(value: boolean): void;
        /**
         * Returns whether the body is a sensor.  Sensor bodies act as normal bodies in the sense that they notify about contacts, however they do not collide with other bodies (they act like ghost bodies).
         *
         * @return   {boolean} if {@code true} the body is a sensor.  It will not collide when enters contact with other bodies
         */
        isSensor(): boolean;
        /**
         * Set whether the body is static.  Static bodies do not move or rotate, unless done manually using {@link #setPosition(float,float) setPosition} or {@link #setRotation(float) setRotation}.
         *
         * @param {boolean} value   if {@code true} the body will be static
         */
        setStaticBody(value: boolean): void;
        /**
         * Set whether the body is static.  Static bodies do not move or rotate, unless done manually using {@link #setPosition(float,float) setPosition} or {@link #setRotation(float) setRotation}.
         *
         * @param {boolean} value   if {@code true} the body will be static
         */
        setStatic(value: boolean): void;
        /**
         * Returns the mass of the body.  Static bodies or bodies not added to the world return 0.
         *
         * @return   {number} the mass of the body or 0 if static
         */
        getMass(): number;
        /**
         * Returns whether the body is static.  Static bodies do not move or rotate, unless done manually using {@link #setPosition(float,float) setPosition} or {@link #setRotation(float) setRotation}.
         *
         * @return   {boolean} if {@code true} the body is static
         */
        isStatic(): boolean;
        /**
         * Set whether the body is a bullet.  Bullet bodies are computationally more expensive but more accurate in their movement.  Use this only with fast objects.
         *
         * @param {boolean} value   if {@code true} the body will be a bullet
         */
        setBullet(value: boolean): void;
        /**
         * Set the restitution of the body.  The restitution determines the ratio of the reaction force normal to a contact, when the body collides with another body.  Basically it can be seen as a coefficient that will control the strength with which the body bounces back from a collision.  The resititution of a contact of two bodies in a collision is calculated as the maximum of the restitution values of the 2 bodies involved.
         *
         * @param {number} restitution   a positive value.  A value of 0 means no bounce after a collision, and a value of 1 means bounce with it's full speed from a collision
         */
        setRestitution(restitution: number): void;
        /**
         * Set the friction of the body.  The friction determines the ratio of the reaction force tangent to a contact, when the body collides with another body.  Basically it can be seen as a coefficient that will control how the body gets slown down when the body slides against another body.  The friction of a contact of two bodies in a collision is calculated from the friction values of the 2 bodies involved.
         *
         * @param {number} friction   a positive value.  A value of 0 means no friction and thus the body will not be slown down if no other forces are applied
         */
        setFriction(friction: number): void;
        /**
         * Set whether the body can rotate.
         *
         * @param {boolean} rotatable   if {@code true} the body will not rotate
         */
        setRotatable(rotatable: boolean): void;
        /**
         * Set whether the body can sleep.
         *
         * @param {boolean} allowSleep if {@code true} the body will be able to sleep
         */
        setAllowSleeping(allowSleep: boolean): void;
        /**
         * Return a list of bodies currently touching the body.
         *
         * @return   {Array} list of bodies (ArrayList of FBody) touching the body
         */
        getTouching(): Array<any>;
        /**
         * Return a list of contacts currently involving the body.
         *
         * @return   {Array} list of contacts (ArrayList of FContact) touching the body
         */
        getContacts(): Array<any>;
        /**
         * Returns a list with all the joints with a connection to the body
         *
         * @return    {Array} an ArrayList (of FJoint) connected to the body
         */
        getJoints(): Array<any>;
        /**
         * Returns true if the body is joint to the body passed as argument
         *
         * @param {fisica.FBody} other  the other body
         * @return     {boolean} if {@code true} the body is connected to other
         */
        isConnected(other: FBody): boolean;
        /**
         * Return whether the body is currently touching the body passed as argument.
         *
         * @param {fisica.FBody} b  the body for which we want to know if there is contact
         * @return   {boolean} if {@code true} the body is touching b
         */
        isTouchingBody(b: FBody): boolean;
        getLocalWorldPoint(p: org.jbox2d.common.Vec2): org.jbox2d.common.Vec2;
        constructor();
    }
}
declare namespace fisica {
    /**
     * Represents a joint between two or more bodies.
     *
     * A joint establishes some kind of relation between two or more bodies.  Depending on the specific joint the relation might be a distance relation, a rotation relation or even a volume conservation relation.  This class cannot be be instantiated, instead use one of the derived classes.
     * @class
     * @extends fisica.FDrawable
     */
    abstract class FJoint extends fisica.FDrawable {
        m_joint: org.jbox2d.dynamics.joints.Joint;
        m_world: fisica.FWorld;
        m_collideConnected: boolean;
        /**
         * WARNING: This method is internal only and may change someday.  If you are using this method please contact the developer since there should be a better way or we may add something to the library.
         *
         * @return {org.jbox2d.dynamics.joints.Joint} the internal JBox2D joint
         */
        getBox2dJoint(): org.jbox2d.dynamics.joints.Joint;
        processJoint(world: org.jbox2d.dynamics.World, jd: org.jbox2d.dynamics.joints.JointDef): void;
        addToWorld(world: fisica.FWorld): void;
        removeFromWorld(): void;
        getJointDef(world: fisica.FWorld): org.jbox2d.dynamics.joints.JointDef;
        preDraw(applet: PGraphics): void;
        postDraw(applet: PGraphics): void;
        preDrawDebug(applet: PGraphics): void;
        postDrawDebug(applet: PGraphics): void;
        /**
         *
         * Returns the first body attached to this joint.
         * @return {fisica.FBody} first of the bodies connected by this joint
         */
        getBody1(): fisica.FBody;
        /**
         *
         * Returns the second body attached to this joint.
         * @return {fisica.FBody} second of the bodies connected by this joint
         */
        getBody2(): fisica.FBody;
        /**
         *
         * Sets whether the bodies connected by the joint should collide with each other.
         *
         * @param {boolean} value  if {@code true} the bodies connected by this joint will be allowed to collide with each other
         */
        setCollideConnected(value: boolean): void;
        /**
         * Returns the horizontal component of the reaction force on the second body at the joint anchor.
         * @return {number} horizontal component of the reaction force
         */
        getReactionForceX(): number;
        /**
         *
         * Returns the vertical component of the reaction force on the second body at the joint anchor.
         * @return {number} vertical component of the reaction force
         */
        getReactionForceY(): number;
        /**
         *
         * Returns the reaction torque on the second body at the joint anchor.
         * @return {number} reaction torque
         */
        getReactionTorque(): number;
        constructor();
    }
}
declare namespace fisica {
    class FAddBodyAction extends fisica.FWorldAction {
        m_body: fisica.FBody;
        constructor(body: fisica.FBody);
        apply(world: fisica.FWorld): void;
    }
}
declare namespace fisica {
    class FRemoveBodyAction extends fisica.FWorldAction {
        m_body: fisica.FBody;
        constructor(body: fisica.FBody);
        apply(world: fisica.FWorld): void;
    }
}
declare namespace fisica {
    class FRemoveJointAction extends fisica.FWorldAction {
        m_joint: fisica.FJoint;
        constructor(joint: fisica.FJoint);
        apply(world: fisica.FWorld): void;
    }
}
declare namespace fisica {
    class FAddJointAction extends fisica.FWorldAction {
        m_joint: fisica.FJoint;
        constructor(joint: fisica.FJoint);
        apply(world: fisica.FWorld): void;
    }
}
declare namespace fisica {
    class FCompound extends fisica.FBody {
        m_shapes: Array<any>;
        constructor();
        getShapeDefs(): Array<any>;
        getBodies(): Array<any>;
        addBody(body: fisica.FBody): void;
        draw(applet: PGraphics): void;
        drawDebug(applet: PGraphics): void;
    }
}
declare namespace fisica {
    /**
     * Constructs a circular body that can be added to a world.
     *
     * @param {number} size  the size of the circle
     * @class
     * @extends fisica.FBody
     */
    class FCircle extends fisica.FBody {
        m_size: number;
        getShapeDef(): org.jbox2d.collision.shapes.ShapeDef;
        getTransformedShapeDef(): org.jbox2d.collision.shapes.ShapeDef;
        constructor(size: number);
        /**
         * Returns the size of the circle.
         *
         * @usage Bodies
         * @return {number} the size of the circle
         */
        getSize(): number;
        /**
         * Sets the size of the circle.
         * Under the hood the body is removed and readded to the world.
         *
         * @usage Bodies
         * @param {number} size the size of the circle
         */
        setSize(size: number): void;
        draw(applet: PGraphics): void;
        drawDebug(applet: PGraphics): void;
    }
}
declare namespace fisica {
    /**
     * Constructs a line body that can be added to a world.
     *
     * @param {number} x1  horizontal position of the first point of the line
     * @param {number} y1  vertical position of the first point of the line
     * @param {number} x2  horizontal position of the second point of the line
     * @param {number} y2  vertical position of the second point of the line
     * @class
     * @extends fisica.FBody
     */
    class FLine extends fisica.FBody {
        m_start: org.jbox2d.common.Vec2;
        m_end: org.jbox2d.common.Vec2;
        getShapeDef(): org.jbox2d.collision.shapes.ShapeDef;
        constructor(x1: number, y1: number, x2: number, y2: number);
        /**
         * Sets the start point of the line.
         * Under the hood the body is removed and readded to the world.
         *
         * @usage Bodies
         * @param {number} x  horizontal position of the first point of the line
         * @param {number} y  vertical position of the first point of the line
         */
        setStart(x: number, y: number): void;
        /**
         * Sets the end point of the line.
         * Under the hood the body is removed and readded to the world.
         *
         * @usage Bodies
         * @param {number} x  horizontal position of the first point of the line
         * @param {number} y  vertical position of the first point of the line
         */
        setEnd(x: number, y: number): void;
        draw(applet: PGraphics): void;
        drawDebug(applet: PGraphics): void;
    }
}
declare namespace fisica {
    /**
     * Constructs a blob body that can be added to a world. It creates an empty
     * blob, before adding the blob to the world use {@link #vertex(float,float)
     * vertex} or {@link #setAsCircle(float) setAsCircle} to define the initial
     * shape of the blob.
     * @class
     * @extends fisica.FBody
     */
    class FBlob extends fisica.FBody {
        m_vertices: Array<any>;
        m_vertexBodies: Array<any>;
        m_damping: number;
        m_frequency: number;
        m_vertexSize: number;
        m_force: org.jbox2d.common.Vec2;
        m_torque: number;
        m_density: number;
        m_restitution: number;
        m_friction: number;
        m_bullet: boolean;
        m_joint: fisica.FConstantVolumeJoint;
        constructor();
        addToWorld(world: fisica.FWorld): void;
        removeFromWorld(): void;
        /**
         * Adds a vertex body to the initial shape of the blob. This method must be
         * called before adding the body to the world.
         *
         * @param {fisica.FBody} b
         * b the body to be added
         */
        addVertexBody(b: fisica.FBody): void;
        /**
         * Adds a vertex to the initial shape of the blob. This method must be called
         * before adding the body to the world.
         *
         * @param {number} x
         * x coordinate of the vertex to be added
         * @param {number} y
         * y coordinate of the vertex to be added
         */
        vertex(x: number, y: number): void;
        /**
         * Gets the x coordinate of the ith vertex of the initial shape of the blob.
         *
         * @param {number} i
         * index of the vertex to retrieve
         * @return {number} the x coordinate of the vertex to retrieve
         */
        getVertexX(i: number): number;
        /**
         * Gets the y coordinate of the ith vertex of the initial shape of the blob.
         *
         * @param {number} i
         * index of the vertex to retrieve
         * @return {number} the y coordinate of the vertex to retrieve
         */
        getVertexY(i: number): number;
        setAsCircle$float$float$float$int(x: number, y: number, size: number, vertexCount: number): void;
        /**
         * Sets the initial shape of the blob to a circle. This method removes all the
         * previous vertices tha may have been added by the use of the
         * {@link #vertex(float,float) vertex}. This method must be called before adding
         * the body to the world.
         *
         * @param {number} x
         * x coordinate of the position of the circle
         * @param {number} y
         * y coordinate of the position of the circle
         * @param {number} size
         * size of the circle
         * @param {number} vertexCount
         * number of vertices of the circle
         */
        setAsCircle(x?: any, y?: any, size?: any, vertexCount?: any): any;
        setAsCircle$float$float$float(x: number, y: number, size: number): void;
        setAsCircle$float(size: number): void;
        setAsCircle$float$int(size: number, vertexCount: number): void;
        /**
         * Returns the size of the circular vertices of the blob. This method must be
         * called before the body is added to the world.
         *
         * @return {number} size of the circular vertices of the blob
         */
        getVertexSize(): number;
        /**
         * Sets the size of the circular vertices of the blob. This method must be
         * called before the body is added to the world.
         *
         * @param {number} size
         * size of the circular vertices of the blob
         */
        setVertexSize(size: number): void;
        /**
         * Returns vertices of the blob.
         *
         * @return {Array} list of vertex bodies
         */
        getVertexBodies(): Array<any>;
        /**
         * Sets the frequency of the springs used to maintain the volume defined by the
         * vertices constant.
         *
         * @param {number} frequency
         * the frequency of the springs of the constant volume joint
         */
        setFrequency(frequency: number): void;
        /**
         * Sets the damping of the springs used to maintain the volume defined by the
         * vertices constant.
         *
         * @param {number} damping
         * the damping of the springs of the constant volume joint
         */
        setDamping(damping: number): void;
        addForce(fx?: any, fy?: any, px?: any, py?: any): any;
        addForce$float$float(fx: number, fy: number): void;
        addTorque(t: number): void;
        setDensity(d: number): void;
        setFriction(d: number): void;
        setRestitution(d: number): void;
        setBullet(d: boolean): void;
        setNoStroke(): void;
        setNoFill(): void;
        setFillColor(col: number): void;
        setStrokeColor(col: number): void;
        setStrokeWeight(col: number): void;
        setDrawable(val: boolean): void;
        attachImage(img: PImage): void;
        dettachImage(): void;
    }
}
declare namespace fisica {
    /**
     * Constructs a polygonal body that can be added to a world.  It creates an empty polygon, before adding the blob to the world use {@link #vertex(float,float) vertex} to define the shape of the polygon.
     * @class
     * @extends fisica.FBody
     */
    class FPoly extends fisica.FBody {
        m_polygon: org.jbox2d.util.nonconvex.Polygon;
        m_closed: boolean;
        m_vertices: Array<any>;
        constructor();
        /**
         * Adds vertices to the shape of the poly.  This method must called before adding the body to the world.
         *
         * @param {number} x  x coordinate of the vertex to be added
         * @param {number} y  y coordinate of the vertex to be added
         */
        vertex(x: number, y: number): void;
        processBody(bd: org.jbox2d.dynamics.Body, sd: org.jbox2d.collision.shapes.ShapeDef): void;
        getShapeDef(): org.jbox2d.collision.shapes.ShapeDef;
        getTransformedShapeDef(): org.jbox2d.collision.shapes.ShapeDef;
        draw(applet: PGraphics): void;
        drawDebug(applet: PGraphics): void;
    }
}
declare namespace fisica {
    /**
     * Constructs a rectangular body that can be added to a world.
     *
     * @param {number} width  the width of the rectangle
     * @param {number} height  the height of the rectangle
     * @class
     * @extends fisica.FBody
     */
    class FBox extends fisica.FBody {
        m_height: number;
        m_width: number;
        getShapeDef(): org.jbox2d.collision.shapes.ShapeDef;
        getTransformedShapeDef(): org.jbox2d.collision.shapes.ShapeDef;
        constructor(width: number, height: number);
        /**
         * Returns the height of the rectangle.
         *
         * @usage Bodies
         * @see #getWidth()
         * @return {number} the height of the rectangle
         */
        getHeight(): number;
        /**
         * Returns the width of the rectangle.
         *
         * @usage Bodies
         * @see #getHeight()
         * @return {number} the width of the rectangle
         */
        getWidth(): number;
        /**
         * Sets the height of the rectangle.
         * Under the hood the body is removed and readded to the world.
         *
         * @usage Bodies
         * @see #getWidth()
         * @return {void} the height of the rectangle
         * @param {number} height
         */
        setHeight(height: number): void;
        /**
         * Sets the width of the rectangle.
         * Under the hood the body is removed and readded to the world.
         *
         * @usage Bodies
         * @see #getWidth()
         * @return {void} the width of the rectangle
         * @param {number} width
         */
        setWidth(width: number): void;
        draw(applet: PGraphics): void;
        drawDebug(applet: PGraphics): void;
    }
}
declare namespace fisica {
    /**
     * Construct a prismatic joint between two bodies.  The constructor automatically sets the anchor of the joint to the center of the second body.
     *
     * @param {fisica.FBody} body1  first body of the joint
     * @param {fisica.FBody} body2  second body of the joint
     * @class
     * @extends fisica.FJoint
     */
    class FPrismaticJoint extends fisica.FJoint {
        m_body1: fisica.FBody;
        m_body2: fisica.FBody;
        m_anchor: org.jbox2d.common.Vec2;
        m_axis: org.jbox2d.common.Vec2;
        m_localAxis1: org.jbox2d.common.Vec2;
        m_force: number;
        m_torque: number;
        m_motorForce: number;
        m_limitForce: number;
        m_limitPositionImpulse: number;
        m_maxMotorForce: number;
        /**
         * The local anchor point relative to body1's origin.
         */
        m_localAnchor1: org.jbox2d.common.Vec2;
        /**
         * The local anchor point relative to body2's origin.
         */
        m_localAnchor2: org.jbox2d.common.Vec2;
        /**
         * The body2 angle minus body1 angle in the reference state (radians).
         */
        m_referenceAngle: number;
        /**
         * A flag to enable joint limits.
         */
        m_enableLimit: boolean;
        /**
         * The lower translation for the joint limit (world coords).
         */
        m_lowerTranslation: number;
        /**
         * The upper translation for the joint limit (world coords).
         */
        m_upperTranslation: number;
        /**
         * A flag to enable the joint motor.
         */
        m_enableMotor: boolean;
        /**
         * The desired motor speed. Usually in radians per second.
         */
        m_motorSpeed: number;
        /**
         * The maximum motor torque used to achieve the desired motor speed.
         * Usually in N-m.
         */
        m_maxMotorTorque: number;
        updateLocalAnchors(): void;
        updateLocalAxis(): void;
        getJointDef(world: fisica.FWorld): org.jbox2d.dynamics.joints.JointDef;
        constructor(body1: fisica.FBody, body2: fisica.FBody);
        /**
         * Sets the axis of movement of the joint.  This is only axis alog which the bodies can translate with relation to each other.  The axis is given global coordinates, relative to the center of the canvas.  This property must be set before adding the joint to the world.
         *
         * @param {number} x  the horizontal component of the axis in global coordinates, relative to the center of the canvas
         * @param {number} y  the vertical coordinate of the axis in global coordinates, relative to the center of the canvas
         */
        setAxis(x: number, y: number): void;
        /**
         * Sets the position of the anchor of the joint.  This position is given global coordinates, relative to the center of the canvas.
         *
         * @param {number} x  the horizontal coordinate of the anchor in global coordinates, relative to the center of the canvas
         * @param {number} y  the vertical coordinate of the anchor in global coordinates, relative to the center of the canvas
         */
        setAnchor(x: number, y: number): void;
        /**
         * Get the horizontal coordinate of the anchor of the joint.  This position is given global coordinates, relative to the center of the canvas.
         *
         * @return  {number} the horizontal coordinate of the anchor in global coordinates, relative to the center of the canvas
         */
        getAnchorX(): number;
        /**
         * Get the vertical coordinate of the anchor of the joint.  This position is given global coordinates, relative to the center of the canvas.
         *
         * @return  {number} the vertical coordinate of the anchor in global coordinates, relative to the center of the canvas
         */
        getAnchorY(): number;
        /**
         * Set the lowest translation allowed.  This property only has effect if the {@code enableLimit} has been set to {@code true} using {@link #setEnableLimit(boolean)}.
         *
         * @param {number} translation  lowest translation position in pixels
         */
        setLowerTranslation(translation: number): void;
        /**
         * Set the highest translation allowed.  This property only has effect if the {@code enableLimit} has been set to {@code true} using {@link #setEnableLimit(boolean)}.
         *
         * @param {number} translation  highest translation position in pixels
         */
        setUpperTranslation(translation: number): void;
        /**
         * Set limits to the allowed translation of one body respect to the other.  If set to {@code true} the limits imposed using {@link #setLowerTranslation(float) setLowerTranslation} and {@link #setUpperTranslation(float) setLowerTranslation} are enforced.
         *
         * @param {boolean} value  if {@code true} the bodies will be able to translate along the axis only between certain limits
         */
        setEnableLimit(value: boolean): void;
        draw(applet: PGraphics): void;
        drawDebug(applet: PGraphics): void;
    }
}
declare namespace fisica {
    /**
     * Construct a distance joint between two bodies.  The constructor automatically sets the anchors of the joint to the centers of each body, and the length of the joint to the current distance between the bodies.
     *
     * @param {fisica.FBody} body1  first body of the joint
     * @param {fisica.FBody} body2  second body of the joint
     * @class
     * @extends fisica.FJoint
     */
    class FDistanceJoint extends fisica.FJoint {
        m_body1: fisica.FBody;
        m_body2: fisica.FBody;
        m_damping: number;
        m_frequency: number;
        m_length: number;
        m_anchor1: org.jbox2d.common.Vec2;
        m_anchor2: org.jbox2d.common.Vec2;
        getJointDef(world: fisica.FWorld): org.jbox2d.dynamics.joints.JointDef;
        constructor(body1: fisica.FBody, body2: fisica.FBody);
        /**
         * Sets the damping of the spring used to maintain the distance between the bodies constant.
         *
         * @param {number} damping  the damping of the spring
         */
        setDamping(damping: number): void;
        /**
         * Sets the frequency of the spring used to maintain the distance between the bodies constant.
         *
         * @param {number} frequency  the frequency of the spring
         */
        setFrequency(frequency: number): void;
        /**
         * Sets the length of the joint to the current distance between the bodies.
         */
        calculateLength(): void;
        /**
         * Sets the target distance for the joint.  This is the distance that the joint will try to maintain between the two bodies.  If you want to use as length the current distance between the two bodies, use the method {@link #calculateLength() calculateLength}.
         *
         * @param {number} length  the length of the joint
         */
        setLength(length: number): void;
        /**
         * Sets the position of the anchor of the second end of the joint on the second body.  This position must be given relative to the center of the second body.  The anchor point is the point used to calculate the distance and to apply forces in order to move the body.
         *
         * @param {number} x  the horizontal coordinate of the second anchor relative to the center of the second body
         * @param {number} y  the vertical coordinate of the second anchor relative to the center of the second body
         */
        setAnchor2(x: number, y: number): void;
        /**
         * Sets the position of the anchor of the first end of the joint on the first body.  This position must be given relative to the center of the first body.  The anchor point is the point used to calculate the distance and to apply forces in order to move the body.
         *
         * @param {number} x  the horizontal coordinate of the first anchor relative to the center of the first body
         * @param {number} y  the vertical coordinate of the first anchor relative to the center of the first body
         */
        setAnchor1(x: number, y: number): void;
        /**
         * Get the horizontal coordinate of the first anchor point on the first body.  This position is given relative to the center of the first body.
         *
         * @return  {number} the horizontal coordinate of the first anchor relative to the center of the first body
         */
        getAnchor1X(): number;
        /**
         * Get the vertical coordinate of the first anchor point on the first body.  This position is given relative to the center of the first body.
         *
         * @return  {number} the vertical coordinate of the first anchor relative to the center of the first body
         */
        getAnchor1Y(): number;
        /**
         * Get the horizontal coordinate of the second anchor point on the second body.  This position is given relative to the center of the second body.
         *
         * @return  {number} the horizontal coordinate of the second anchor relative to the center of the second body
         */
        getAnchor2X(): number;
        /**
         * Get the vertical coordinate of the second anchor point on the second body.  This position is given relative to the center of the second body.
         *
         * @return  {number} the vertical coordinate of the second anchor relative to the center of the second body
         */
        getAnchor2Y(): number;
        draw(applet: PGraphics): void;
        drawDebug(applet: PGraphics): void;
    }
}
declare namespace fisica {
    /**
     * Construct a gear joint between two joints.
     *
     * @param {fisica.FJoint} joint1  first joint of the gear
     * @param {fisica.FJoint} joint2  second joint of the gear
     * @class
     * @extends fisica.FJoint
     */
    class FGearJoint extends fisica.FJoint {
        m_joint1: fisica.FJoint;
        m_joint2: fisica.FJoint;
        m_ratio: number;
        m_worldRatio: number;
        getJointDef(world: fisica.FWorld): org.jbox2d.dynamics.joints.JointDef;
        updateRatio(): void;
        constructor(joint1: fisica.FJoint, joint2: fisica.FJoint);
        /**
         * Sets the ratio of movement transfer from one joint to the other of the gear.
         *
         * @param {number} ratio  the ratio of movement that is transfered between the first and the second joints of the gear
         */
        setRatio(ratio: number): void;
        draw(applet: PGraphics): void;
    }
}
declare namespace fisica {
    /**
     * Construct a revolute joint between two bodies given an anchor position.
     *
     * @param {fisica.FBody} body1  first body of the joint
     * @param {fisica.FBody} body2  second body of the joint
     * @param {number} x  horizontal coordinate of the anchor given in global coordinates, relative to the canvas' center
     * @param {number} y  vertical coordinate of the anchor given in global coordinates, relative to the canvas' center
     * @class
     * @extends fisica.FJoint
     */
    class FRevoluteJoint extends fisica.FJoint {
        m_body1: fisica.FBody;
        m_body2: fisica.FBody;
        m_anchor: org.jbox2d.common.Vec2;
        /**
         * The local anchor point relative to body1's origin.
         */
        m_localAnchor1: org.jbox2d.common.Vec2;
        /**
         * The local anchor point relative to body2's origin.
         */
        m_localAnchor2: org.jbox2d.common.Vec2;
        /**
         * The body2 angle minus body1 angle in the reference state (radians).
         */
        m_referenceAngle: number;
        /**
         * A flag to enable joint limits.
         */
        m_enableLimit: boolean;
        /**
         * The lower angle for the joint limit (radians).
         */
        m_lowerAngle: number;
        /**
         * The upper angle for the joint limit (radians).
         */
        m_upperAngle: number;
        /**
         * A flag to enable the joint motor.
         */
        m_enableMotor: boolean;
        /**
         * The desired motor speed. Usually in radians per second.
         */
        m_motorSpeed: number;
        /**
         * The maximum motor torque used to achieve the desired motor speed.
         * Usually in N-m.
         */
        m_maxMotorTorque: number;
        updateLocalAnchors(): void;
        getJointDef(world: fisica.FWorld): org.jbox2d.dynamics.joints.JointDef;
        constructor(body1?: any, body2?: any, x?: any, y?: any);
        /**
         * Set the lowest angle allowed.  This property only has effect if the {@code enableLimit} has been set to {@code true} using {@link #setEnableLimit(boolean)}.
         *
         * @param {number} a  lowest angle allowed in radians
         */
        setLowerAngle(a: number): void;
        /**
         * Set the highest angle allowed.  This property only has effect if the {@code enableLimit} has been set to {@code true} using {@link #setEnableLimit(boolean)}.
         *
         * @param {number} a  highest angle allowed in radians
         */
        setUpperAngle(a: number): void;
        /**
         * Set limits to the allowed rotation of one body respect to the other.  If set to {@code true} the limits imposed using {@link #setLowerAngle(float) setLowerAngle} and {@link #setUpperAngle(float) setLowerAngle} are enforced.
         *
         * @param {boolean} value  if {@code true} the bodies will be able to rotate around the anchor only between certain limits
         */
        setEnableLimit(value: boolean): void;
        /**
         * Set the desired rotation speed of the joint.  This property only has effect if the {@code enableMotor} has been set to {@code true} using {@link #setEnableMotor(boolean)}.  The speed is given in radians per second.
         *
         * @param {number} a  the desired speed in radians per second
         */
        setMotorSpeed(a: number): void;
        /**
         * Set the maximum torque that the joint's motor can apply in order to acheive the desired speed.  This property only has effect if the {@code enableMotor} has been set to {@code true} using {@link #setEnableMotor(boolean)}.
         *
         * @param {number} a  the maximum torque of the joint's motor
         */
        setMaxMotorTorque(a: number): void;
        /**
         * Set the state of the motor in order to generate a rotation force (torque) on the joint.  If set to {@code true} the desired motor speed, set using {@link #setMotorSpeed(float) setMotorSpeed}, will try to be matched using a motor with a maximum rotation force (torque) set using {@link #setMaxMotorTorque(float) setMaxMotorTorque}.
         *
         * @param {boolean} value  if {@code true} the joint will receive the rotation force (torque) of a motor
         */
        setEnableMotor(value: boolean): void;
        /**
         * Sets the position of the anchor of the joint around which the bodies rotate.  This position is given global coordinates, relative to the center of the canvas.
         *
         * @param {number} x  the horizontal coordinate of the anchor in global coordinates, relative to the center of the canvas
         * @param {number} y  the vertical coordinate of the anchor in global coordinates, relative to the center of the canvas
         */
        setAnchor(x: number, y: number): void;
        /**
         * Get the horizontal coordinate of the anchor of the joint around which the bodies can rotate.  This position is given global coordinates, relative to the center of the canvas.
         *
         * @return  {number} the horizontal coordinate of the anchor in global coordinates, relative to the center of the canvas
         */
        getAnchorX(): number;
        /**
         * Get the vertical coordinate of the anchor of the joint around which the bodies can rotate.  This position is given global coordinates, relative to the center of the canvas.
         *
         * @return  {number} the vertical coordinate of the anchor in global coordinates, relative to the center of the canvas
         */
        getAnchorY(): number;
        setReferenceAngle(ang: number): void;
        draw(applet: PGraphics): void;
        drawDebug(applet: PGraphics): void;
    }
}
declare namespace fisica {
    /**
     * Construct a mouse joint between a body and a target.  The constructor automatically sets the anchors of the joint to the center of the body.
     *
     * @param {fisica.FBody} body  the body to be grabbed by the joint
     * @param {number} x  horizontal coordinate of the initial target of the joint
     * @param {number} y  vertical coordinate of the initial target of the joint
     * @class
     * @extends fisica.FJoint
     */
    class FMouseJoint extends fisica.FJoint {
        m_fbody: fisica.FBody;
        m_anchor: org.jbox2d.common.Vec2;
        m_target: org.jbox2d.common.Vec2;
        m_damping: number;
        m_frequency: number;
        getJointDef(world: fisica.FWorld): org.jbox2d.dynamics.joints.JointDef;
        constructor(body: fisica.FBody, x: number, y: number);
        /**
         * Sets the damping of the spring used to maintain the body and the target together.  This property must be set before adding the joint to the world.
         *
         * @param {number} damping  the damping of the spring
         */
        setDamping(damping: number): void;
        /**
         * Sets the frequency of the spring used to maintain the body and the target together.  This property must be set before adding the joint to the world.
         *
         * @param {number} frequency  the frequency of the spring
         */
        setFrequency(frequency: number): void;
        /**
         * Sets the target position of the joint.  By setting this property everytime the mouse is used we are able to make the body of this joint follow mouse.
         *
         * @param {number} x  horizontal coordinate of the target of the joint
         * @param {number} y  vertical coordinate of the target of the joint
         */
        setTarget(x: number, y: number): void;
        /**
         * Returns the horizontal target position of the joint.
         *
         * @return  {number} horizontal coordinate of the target of the joint
         */
        getTargetX(): number;
        /**
         * Returns the vertical target position of the joint.
         *
         * @return  {number} vertical coordinate of the target of the joint
         */
        getTargetY(): number;
        /**
         * Sets the body grabbed by this joint and the target position.
         *
         * @param {fisica.FBody} body  the body to be grabbed by the joint
         * @param {number} x  horizontal coordinate of the target of the joint
         * @param {number} y  vertical coordinate of the target of the joint
         */
        setGrabbedBodyAndTarget(body: fisica.FBody, x: number, y: number): void;
        /**
         * Releases the body grabbed by this joint.
         */
        releaseGrabbedBody(): void;
        /**
         * Returns the body grabbed by this joint.
         *
         * @return {fisica.FBody} the body grabbed by this joint
         */
        getGrabbedBody(): fisica.FBody;
        /**
         * Sets the anchor position at which the joint grabs the body.  The anchor point is the point used to apply forces in order to move the body.
         *
         * @param {number} x  the horizontal coordinate of the anchor relative to the center of the body
         * @param {number} y  the vertical coordinate of the anchor relative to the center of the body
         */
        setAnchor(x: number, y: number): void;
        /**
         * Get the horizontal coordinate of the anchor point on the body.  This position is given relative to the center of the body.
         *
         * @return  {number} the horizontal coordinate of the anchor relative to the center of the first body
         */
        getAnchorX(): number;
        /**
         * Get the vertical coordinate of the anchor point on the body.  This position is given relative to the center of the body.
         *
         * @return  {number} the vertical coordinate of the anchor relative to the center of the first body
         */
        getAnchorY(): number;
        draw(applet: PGraphics): void;
        drawDebug(applet: PGraphics): void;
    }
}
declare namespace fisica {
    /**
     * Constructs an constant volume joint.  It creates an empty joint, before adding the joint to the world use {@link #addBody(FBody) addBody} to add bodies to the joint and to define the initial and target volume of the joint.
     * @class
     * @extends fisica.FJoint
     */
    class FConstantVolumeJoint extends fisica.FJoint {
        m_bodies: Array<any>;
        m_damping: number;
        m_frequency: number;
        getJointDef(world: fisica.FWorld): org.jbox2d.dynamics.joints.JointDef;
        constructor();
        /**
         * Adds a body to the joint.  This method must be called before adding the joint to the world.
         *
         * @param {fisica.FBody} b  body to be added
         */
        addBody(b: fisica.FBody): void;
        /**
         * Return the group of bodies that are connected by this joint.
         *
         * @return   {Array} list of bodies (ArrayList of FBody) connected by the joint
         */
        getBodies(): Array<any>;
        /**
         * Sets the damping of the springs used to maintain the volume defined by the vertices constant.  This method must be called before adding the joint to the world.
         *
         * @param {number} damping  the damping of the springs of the constant volume joint
         */
        setDamping(damping: number): void;
        /**
         * Sets the frequency of the springs used to maintain the volume defined by the vertices constant.  This method must be called before adding the joint to the world.
         *
         * @param {number} frequency  the frequency of the springs of the constant volume joint
         */
        setFrequency(frequency: number): void;
        getCentroid(): PVector;
        draw(applet: PGraphics): void;
        drawDebug(applet: PGraphics): void;
    }
}
